<?php

namespace Go2B\Controllers;

use Phalcon\Http\Response;
use Phalcon\Mvc\Model\Resultset;
use League\OAuth2\Server\AuthorizationServer;
use League\OAuth2\Server\Exception\OAuthServerException;
use Phalcon\Http\Request\Exception as HttpException;
use Rest\Components\HttpStatusCodes;
use Rest\Components\Request as OAuth2Request;
use Rest\Components\Response as OAuth2Response;

/**
 * Base RESTful Controller.
 * Supports queries with the following parameters:
 *   Searching:
 *     q=(searchField1:value1,searchField2:value2)
 *   Partial Responses:
 *     fields=(field1,field2,field3)
 *   Limits:
 *     limit=10
 *   Partials:
 *     offset=20
 *
 */
class RestController extends ControllerBase {
  /**
   * If query string contains 'q' parameter.
   * This indicates the request is searching an entity
   * @var boolean
   */
  protected $isSearch = false;

  /**
   * If query contains 'fields' parameter.
   * This indicates the request wants back only certain fields from a record
   * @var boolean
   */
  protected $isPartial = false;

  /**
   * Set when there is a 'limit' query parameter
   * @var integer
   */
  protected $limit = null;

  /**
   * Set when there is an 'offset' query parameter
   * @var integer
   */
  protected $offset = null;

  /**
   * Array of fields requested to be searched against
   * @var array
   */
  protected $searchFields = null;

  /**
   * Array of fields requested to be returned
   * @var array
   */
  protected $partialFields = null;

  /**
   * Sets which fields may be searched against, and which fields are allowed to be returned in
   * partial responses.  This will be overridden in child Controllers that support searching
   * and partial responses.
   * @var array
   */
  protected $allowedFields = [
    'search' => [],
    'partials' => []
  ];

  /**
   * Constructor, calls the parse method for the query string by default.
   * @param boolean $parseQueryString true Can be set to false if a controller needs to be called
   *        from a different controller, bypassing the $allowedFields parse
   */
  public function onConstruct($parseQueryString = true) {
    if ($parseQueryString){
      $this->parseRequest($this->allowedFields);
    }
    return;
  }

  /**
   * Parses out the search parameters from a request.
   * Unparsed, they will look like this:
   *    (name:Benjamin Framklin,location:Philadelphia)
   * Parsed:
   *     ['name'=>'Benjamin Franklin', 'location'=>'Philadelphia']
   * @param  string $unparsed Unparsed search string
   * @return array            An array of fieldname=>value search parameters
   */
  protected function parseSearchParameters($unparsed) {
    // Strip parentheses that come with the request string
    $unparsed = trim($unparsed, '()');

    // Now we have an array of "key:value" strings.
    $splitFields = explode(',', $unparsed);
    $mapped = [];

    // Split the strings at their colon, set left to key, and right to value.
    foreach ($splitFields as $field) {
      $splitField = explode(':', $field);
      $mapped[$splitField[0]] = $splitField[1];
    }

    return $mapped;
  }

  /**
   * Parses out partial fields to return in the response.
   * Unparsed:
   *     (id,name,location)
   * Parsed:
   *     ['id', 'name', 'location']
   * @param  string $unparsed Un-parsed string of fields to return in partial response
   * @return array            Array of fields to return in partial response
   */
  protected function parsePartialFields($unparsed) {
    return explode(',', trim($unparsed, '()'));
  }

  /**
   * Main method for parsing a query string.
   * Finds search parameters, partial response fields, limits, and offsets.
   * Sets Controller fields for these variables.
   *
   * @param  array $allowedFields Allowed fields array for search and partials
   * @return boolean              Always true if no exception is thrown
   */
  protected function parseRequest($allowedFields) {
    $request = $this->request;
    $searchParams = $request->get('q', null, null);
    $fields = $request->get('fields', null, null);

    // Set limits and offset, elsewise allow them to have defaults set in the Controller
    $this->limit = ($request->get('limit', null, null)) ?: $this->limit;
    $this->offset = ($request->get('offset', null, null)) ?: $this->offset;

    // If there's a 'q' parameter, parse the fields, then determine that all the fields in the search
    // are allowed to be searched from $allowedFields['search']
    if ($searchParams) {
      $this->isSearch = true;
      $this->searchFields = $this->parseSearchParameters($searchParams);

      // This handy snippet determines if searchFields is a strict subset of allowedFields['search']
      if (array_diff(array_keys($this->searchFields), $this->allowedFields['search'])) {
        $httpStatusCode = 401;
        $message = 'You requested to return fields that are not available to be searched.';
        $this->sendFailure($httpStatusCode, $message);
      }
    }

    // If there's a 'fields' parameter, this is a partial request.  Ensures all the requested fields
    // are allowed in partial responses.
    if ($fields) {
      $this->isPartial = true;
      $this->partialFields = $this->parsePartialFields($fields);

      // Determines if fields is a strict subset of allowed fields
      if (array_diff($this->partialFields, $this->allowedFields['partials'])) {
        $httpStatusCode = 401;
        $message = 'You requested to return fields that are not available to be returned in partial responses.';
        $this->sendFailure($httpStatusCode, $message);
      }
    }

    return true;
  }

  public function sendSuccess($data) {
    $this->response->setContentType("application/json");
    $this->response->setStatusCode(200, HttpStatusCodes::getMessage(200))->sendHeaders();
    echo $data;
  }

  public function sendFailure($httpStatusCode, $message) {
    $this->response->setContentType("application/json");
    $this->response->setStatusCode($httpStatusCode, HttpStatusCodes::getMessage($httpStatusCode))->sendHeaders();
    echo json_encode($message);
  }

  //region My API
  public function loginAction() {
    $this->view->disable();

    $oAuth2Request  = new OAuth2Request($this->request);
    $oAuth2Response = new OAuth2Response($this->response);

    try {
      $response = $this->authorizationServer->respondToAccessTokenRequest($oAuth2Request, $oAuth2Response);

      $stream   = $response->getStream();
      $token    = $stream->getToken();

      $this->sendSuccess($token);
    } catch (\Exception $exception) {
      $this->sendFailure($exception->getHttpStatusCode(), $exception->getMessage());
    }
  }

  public function getCatalogsAction() {
    $this->view->disable();
    $this->logger->info('RestController getCatalogsAction');
    $this->logger->info(print_r('RestController $isSearch', true));
    $this->logger->info(print_r($this->isSearch, true));
    $this->logger->info(print_r('RestController $isPartial', true));
    $this->logger->info(print_r($this->isPartial, true));
    $this->logger->info(print_r('RestController $limit', true));
    $this->logger->info(print_r($this->limit, true));
    $this->logger->info(print_r('RestController $offset', true));
    $this->logger->info(print_r($this->offset, true));
    $this->logger->info(print_r('RestController $searchFields', true));
    $this->logger->info(print_r($this->searchFields, true));
    $this->logger->info(print_r('RestController $partialFields', true));
    $this->logger->info(print_r($this->partialFields, true));

    echo json_encode('RestController getCatalogsAction echo json_encode');
  }
  //endregion
}
